// @ts-nocheck
import type { Core } from "@strapi/strapi";
import { UID } from "@strapi/types";
import { contentTypes } from "@strapi/utils";
import pluralize from "pluralize";

interface Options {
  relationalFields?: string[];
}

const { CREATED_BY_ATTRIBUTE, UPDATED_BY_ATTRIBUTE } = contentTypes.constants;
const modelCache = new Map();

const extractPathSegment = (url: string) =>
  url.match(/\/([^/?]+)(?:\?|$)/)?.[1] || "";

const getDeepPopulate = (uid: UID.Schema, opts: Options = {}, depth = 0) => {
  if (depth > 5) return {};
  let model = modelCache.get(uid);
  if (!model) {
    try {
      model = strapi.getModel(uid);
      modelCache.set(uid, model);
    } catch (error) {
      strapi.log.error(`Error getting model ${uid}:`, error);
      return {};
    }
  }

  const acc = {};
  for (const attributeName in model.attributes) {
    const attribute = model.attributes[attributeName];

    switch (attribute.type) {
      case "relation": {
        const isMorphRelation = attribute.relation
          .toLowerCase()
          .startsWith("morph");
        if (isMorphRelation) break;

        const isVisible = contentTypes.isVisibleAttribute(model, attributeName);

        if (isVisible) {
          if (attributeName === "testimonials") {
            acc[attributeName] = { populate: "user.image" };
          } else {
            const relatedUID = attribute.target;
            if (
              relatedUID &&
              depth < 3 &&
              !relatedUID.includes("users-permissions") &&
              !relatedUID.includes("admin::")
            ) {
              acc[attributeName] = {
                populate: getDeepPopulate(relatedUID, opts, depth + 1),
              };
            } else {
              acc[attributeName] = { populate: "*" };
            }
          }
        }
        break;
      }

      case "media": {
        acc[attributeName] = { populate: "*" };
        break;
      }

      case "component": {
        const componentUID = attribute.component;
        if (componentUID && depth < 3) {
          acc[attributeName] = {
            populate: getDeepPopulate(componentUID, opts, depth + 1),
          };
        }
        break;
      }

      case "dynamiczone": {
        const components = attribute.components || [];
        if (components.length > 0 && depth < 3) {
          const populatedComponents = {};

          for (let i = 0; i < components.length; i++) {
            const componentUID = components[i];
            if (componentUID) {
              populatedComponents[componentUID] = {
                populate: getDeepPopulate(componentUID, opts, depth + 1),
              };
            }
          }

          acc[attributeName] = { on: populatedComponents };
        }
        break;
      }
    }
  }

  return acc;
};

export default (config, { strapi }: { strapi: Core.Strapi }) => {
  return async (ctx, next) => {
    if (
      ctx.request.url.startsWith("/api/") &&
      ctx.request.method === "GET" &&
      !ctx.query.populate &&
      !ctx.request.url.includes("/api/users") &&
      !ctx.request.url.includes("/api/seo")
    ) {
      strapi.log.info("Using optimized Dynamic-Zone population Middleware...");

      const contentType = extractPathSegment(ctx.request.url);
      const singular = pluralize.singular(contentType);
      const uid = `api::${singular}.${singular}`;

      const basePopulate = getDeepPopulate(uid);

      ctx.query.populate = {
        ...basePopulate,
        ...(!ctx.request.url.includes("products") && {
          localizations: { populate: {} },
        }),
      };
    }
    await next();
  };
};
