const xsite = require("../../xsite.json");

// Generate a slug from the target field
function generateSlug(targetField) {
  if (!targetField) return "";
  return targetField
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, "")
    .replace(/[\s_-]+/g, "-")
    .replace(/^-+|-+$/g, "");
}

function randomUIDSlugGenerator(entry, schema) {
  const slugField = Object.keys(schema.attributes || {}).find(
    (key) => schema.attributes[key].type === "uid"
  );

  if (!slugField) {
    return entry;
  }

  entry[slugField] = crypto.randomUUID();
  return entry;
}

// Generate a slug from the target field if the slug field is not provided
function autoGenerateSlug(entry, schema) {
  const slugField = Object.keys(schema.attributes || {}).find(
    (key) => schema.attributes[key].type === "uid"
  );

  if (!slugField) {
    return entry;
  }

  const slugConfig = schema.attributes[slugField];
  const targetField = slugConfig.targetField;

  if (entry[slugField]) {
    return entry;
  }

  if (!targetField || !entry[targetField]) {
    return entry;
  }

  entry[slugField] = generateSlug(entry[targetField]);
  return entry;
}

const getCleanVariableNameWithUnderscore = (name) => {
  return name.replace(/[^a-zA-Z0-9_$]/g, "_");
};

function validateXSiteConfig() {
  if (
    !xsite["xsite-api-key"] ||
    !xsite["xsite-api-secret"] ||
    !xsite.url ||
    !xsite.organizationId ||
    !xsite.projectId ||
    !xsite["xsite-strapi-builder-url"]
  ) {
    // check which config is missing
    const missingConfigs = [];
    if (!xsite["xsite-api-key"]) missingConfigs.push("xsite-api-key");
    if (!xsite["xsite-api-secret"]) missingConfigs.push("xsite-api-secret");
    if (!xsite.url) missingConfigs.push("url");
    if (!xsite.organizationId) missingConfigs.push("organizationId");
    if (!xsite.projectId) missingConfigs.push("projectId");
    if (!xsite["xsite-strapi-builder-url"])
      missingConfigs.push("xsite-strapi-builder-url");

    throw new Error(
      `[ERROR] Missing xsite configuration: ${missingConfigs.join(
        ", "
      )}. Please check your xsite.json file.`
    );
  }
}

module.exports = {
  generateSlug,
  randomUIDSlugGenerator,
  autoGenerateSlug,
  getCleanVariableNameWithUnderscore,
  validateXSiteConfig,
};
