"use strict";

const DEFAULT_PERMISSIONS = [
  "plugin::i18n.locales.listLocales",
  "plugin::upload.content-api.find",
  "plugin::upload.content-api.findOne",
];

const CONTENT_API_ACTIONS = ["find", "findOne"];

async function getPublicRole() {
  const publicRole = await strapi
    .query("plugin::users-permissions.role")
    .findOne({
      where: { type: "public" },
    });

  if (!publicRole) {
    throw new Error("Public role not found");
  }

  return publicRole;
}

function getApiContentTypes() {
  return Object.keys(strapi.contentTypes).filter((uid) =>
    uid.startsWith("api::")
  );
}

function generatePermissions(contentTypes) {
  const permissions = [...DEFAULT_PERMISSIONS];

  for (const uid of contentTypes) {
    const apiName = uid.split("::")[1];

    for (const action of CONTENT_API_ACTIONS) {
      permissions.push(`api::${apiName}.${action}`);
    }
  }

  return permissions;
}

async function createPermissionIfNotExists(action, roleId) {
  const existingPermission = await strapi
    .query("plugin::users-permissions.permission")
    .findOne({
      where: { action, role: roleId },
    });

  if (existingPermission) {
    return false;
  }

  await strapi.query("plugin::users-permissions.permission").create({
    data: {
      action,
      role: roleId,
    },
  });

  return true;
}

async function setPublicPermissions() {
  try {
    const publicRole = await getPublicRole();
    const contentTypes = getApiContentTypes();
    const permissions = generatePermissions(contentTypes);

    let createdCount = 0;
    let skippedCount = 0;

    console.log(
      `[INFO] Processing ${permissions.length} permissions for ${contentTypes.length} content types`
    );

    for (const permission of permissions) {
      const wasCreated = await createPermissionIfNotExists(
        permission,
        publicRole.id
      );

      if (wasCreated) {
        createdCount++;
      } else {
        skippedCount++;
      }
    }

    console.log(
      `[INFO] Permissions processed: ${createdCount} created, ${skippedCount} already existed`
    );
    console.log("[INFO] Public permissions setup completed successfully");

    return {
      success: true,
      created: createdCount,
      skipped: skippedCount,
      total: permissions.length,
    };
  } catch (error) {
    console.error("[ERROR] Failed to set public permissions:", error.message);

    return {
      success: false,
      error: error.message,
    };
  }
}

module.exports = setPublicPermissions;
