"use strict";
const fs = require("fs-extra");
const setPublicPermissions = require("./setPublicPermissions");
const { seedData } = require("./content-types");
const { downloadAllFiles } = require("./download");
const xsite = require("../../xsite.json");
const { mapDynamicZoneComponents } = require("./dynamic-zone-mapper");
const { validateXSiteConfig } = require("./utils");

const DATA_DIR = "./data";
const DATA_FILE = "./data/data.json";

async function fetchXSiteData() {
  validateXSiteConfig();

  const url = `${xsite.url}/organization/${xsite.organizationId}/project/${xsite.projectId}/get-json`;

  const response = await fetch(url, {
    method: "GET",
    headers: {
      "xsite-api-key": xsite["xsite-api-key"],
      "xsite-api-secret": xsite["xsite-api-secret"],
    },
  });

  if (!response.ok) {
    throw new Error(
      `Failed to fetch data: ${response.status} ${response.statusText}`
    );
  }

  return await response.json();
}

async function saveDataToFile(data) {
  await fs.outputFile(DATA_FILE, JSON.stringify(data, null, 2));
  console.log("[INFO] Data saved to file");
}

async function importSeedData() {
  try {
    if (!fs.existsSync(DATA_DIR)) {
      fs.mkdirSync(DATA_DIR, { recursive: true });
    }

    console.log("[INFO] Starting seed data import");

    const res = await fetchXSiteData();

    let mapperJSON;
    try {
      const mapperJSONResponse = await fetch(
        `${xsite["xsite-strapi-builder-url"]}/mapper.json`
      );
      mapperJSON = await mapperJSONResponse.json();
    } catch (error) {
      throw error;
    }

    let data;
    try {
      data = mapDynamicZoneComponents(res, mapperJSON);
    } catch (error) {
      throw error;
    }

    await saveDataToFile(data);

    await setPublicPermissions();
    await downloadAllFiles();
    await seedData();

    console.log("[INFO] Seed data import completed");
  } catch (error) {
    throw error;
  }
}

async function initializeStrapi() {
  const { createStrapi, compileStrapi } = require("@strapi/strapi");
  console.log("[INFO] Initializing Strapi application");
  const appContext = await compileStrapi();
  const app = await createStrapi(appContext).load();
  return app;
}

async function main() {
  let app;
  try {
    app = await initializeStrapi();
    await importSeedData();

    console.log("[INFO] Seed process completed successfully");
    process.exit(0);
  } catch (error) {
    console.error("[ERROR] Seed process failed:", error.message);
    process.exit(1);
  } finally {
    if (app) {
      await app.destroy();
    }
  }
}

main().catch((error) => {
  console.error("[ERROR] Unhandled error:", error.message);
  process.exit(1);
});
