const fs = require("fs");
const path = require("path");
const axios = require("axios");
const { checkFileExistsBeforeUpload } = require("./upload-file");
const xsite = require("../../xsite.json");

const DOWNLOAD_FOLDER = path.join(path.resolve(), "data", "uploads");

async function fetchProjectAssets() {
  const { data } = await axios.get(
    `${xsite.url}/organization/${xsite.organizationId}/project/${xsite.projectId}/asset/get/project-assets?all=true`,
    {
      headers: {
        "xsite-api-key": xsite["xsite-api-key"],
        "xsite-api-secret": xsite["xsite-api-secret"],
      },
    }
  );
  return data.data;
}

function generateFilename(file) {
  let filename = file.name || path.basename(new URL(file.url).pathname);
  if (!path.extname(filename)) {
    filename += ".jpg";
  }
  return filename;
}

async function downloadFileFromUrl(url, filePath) {
  const response = await axios({
    method: "GET",
    url,
    responseType: "stream",
  });

  const writer = fs.createWriteStream(filePath);
  response.data.pipe(writer);

  return new Promise((resolve, reject) => {
    writer.on("finish", resolve);
    writer.on("error", reject);
  });
}

async function downloadFile(file) {
  try {
    const filename = generateFilename(file);
    const filePath = path.join(DOWNLOAD_FOLDER, filename);

    if (!fs.existsSync(filePath)) {
      await downloadFileFromUrl(file.url, filePath);
    }

    return await checkFileExistsBeforeUpload([filename], file?.alternativeText);
  } catch (error) {
    console.warn(`[WARN] Failed to process file ${file.url}`);
  }
}

async function downloadAllFiles() {
  try {
    if (!fs.existsSync(DOWNLOAD_FOLDER)) {
      fs.mkdirSync(DOWNLOAD_FOLDER, { recursive: true });
    }

    const files = await fetchProjectAssets();
    if (!files.length) {
      console.log("[INFO] No files to download");
      return;
    }

    console.log(`[INFO] Starting download of ${files.length} files`);

    for (const file of files) {
      await downloadFile(file);
    }

    console.log("[INFO] All downloads complete");
  } catch (error) {
    console.error(`[ERROR] Download process failed: ${error.message}`);
    throw error;
  }
}

module.exports = {
  downloadAllFiles,
  downloadFile,
};
